<?php

namespace Modules\Finance\App\DataTables;

use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Modules\Finance\App\Enums\DepositEnum;
use Modules\Finance\App\Models\Deposit;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class PendingDepositsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {

        return (new EloquentDataTable($query))
            ->addColumn('action', function ($query) {
                $button = '<div class="d-flex align-items-center">';
                $button .= '<a href="javascript:void(0)" class="btn btn-navy d-flex align-items-center justify-content-center py-2 rounded-3 w-px-40 m-1 deposit-confirm" action-type="confirm" title="' . localize("Confirm") . '" data-action="' . route('deposit.update', ['deposit' => $query->id, 'set_status' => DepositEnum::CONFIRM]) . '"><i class="fa fa-check"></i></a>';
                $button .= '<a href="javascript:void(0)" class="btn btn-red d-flex align-items-center justify-content-center py-2 rounded-3 w-px-40 m-1 deposit-confirm" action-type="cancel" title="' . localize("Cancel") . '" data-action="' . route('deposit.update', ['deposit' => $query->id, 'set_status' => DepositEnum::CANCEL]) . '"><i class="fa fa-ban" aria-hidden="true"></i></a>';
                $button .= '</div>';
                return $button;
            })
            ->editColumn('date', function ($query) {
                return get_ymd($query->date);
            })
            ->editColumn('customer_id', function ($query) {
                return $query->customerInfo['first_name'] . " " . $query->customerInfo['last_name'];
            })
            ->editColumn('accept_currency_id', function ($query) {
                return $query->currencyInfo['symbol'];
            })
            ->editColumn('status', function ($query) {
                return '<span class="badge bg-label-warning py-2"><i class="fa fa-spinner fa-spin"></i> ' . DepositEnum::PENDING->name . '</span>';
            })
            ->setRowId('id')
            ->addIndexColumn()
            ->rawColumns(['action', 'status']);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Deposit $model): QueryBuilder
    {
        return $model->newQuery()
            ->where('status', DepositEnum::PENDING->value)
            ->orderBy('id', 'desc')
            ->with('customerInfo');
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        $tableId = 'deposit-table';
        return $this->builder()
            ->setTableId($tableId)
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0)
            ->dom("<'row m-3'<'col-md-4'l><'col-md-4 text-center'B><'col-md-4'f>>rt<'bottom'<'row'<'col-md-6'i><'col-md-6'p>>><'clear'>")
            ->parameters([
                'responsive'     => true,
                'autoWidth'      => false,
                'headerCallback' => 'function(thead, data, start, end, display) {
                            $(thead).addClass("bg-primary");
                            $(thead).find("th").addClass("text-white");
                            $("#' . $tableId . '").removeClass("table-striped table-hover");
                        }',
            ])
            ->buttons([
                Button::make('excel')->text('<i class="fa fa-file-excel"></i> Excel')->className('btn btn-success box-shadow--4dp btn-sm-menu'),
                Button::make('csv')->text('<i class="fa fa-file-csv"></i> CSV')->className('btn btn-success box-shadow--4dp btn-sm-menu'),
                Button::make('pdf')->text('<i class="fa fa-file-pdf"></i> PDF')->className('btn btn-success box-shadow--4dp btn-sm-menu'),
                Button::make('print')->text('<i class="fa fa-print"></i> Print')->className('btn btn-success box-shadow--4dp btn-sm-menu')->attr(['target' => '_blank']),
                Button::make('reset')->text('<i class="fa fa-undo-alt"></i> Reset')->className('btn btn-success box-shadow--4dp btn-sm-menu'),
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')->title(localize('SL'))->searchable(false)->orderable(false)->width(30)->addClass('text-center'),
            Column::make('date')->title(localize('Date')),
            Column::make('customer_id')->title(localize('Name')),
            Column::make('method')->title(localize('Method')),
            Column::make('amount')->title(localize('Amount')),
            Column::make('accept_currency_id')->title(localize('Coin'))->addClass('text-center'),
            Column::make('fees')->title(localize('Fees'))->addClass('text-center'),
            Column::make('comments')->title(localize('Comments')),
            Column::make('status')->title(localize('Status'))->addClass('text-center'),
            Column::computed('action')->title(localize('Action'))
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),

        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Deposits_' . date('YmdHis');
    }
}
